/*
 * @(#)CERConnectionFigure.java  1.1  2006-02-21
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.cer.diagram;

import ch.hslu.cm.cer.model.*;
import ch.hslu.cm.simulation.*;
import ch.hslu.cm.*;
import java.beans.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.undo.*;
import javax.swing.*;
import java.awt.geom.*;
import org.jhotdraw.app.action.ActionUtil;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.decoration.ArrowTip;
import org.jhotdraw.draw.decoration.GeneralPathTip;
import org.jhotdraw.draw.decoration.LineDecoration;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.layouter.LocatorLayouter;
import org.jhotdraw.draw.locator.BezierLabelLocator;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;

/**
 * CERConnectionFigure.
 * 
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.1 2006-02-21 Support for conditional cardinality added.
 * <br>1.0 2006-01-18 Created.
 */
public class CERConnectionFigure extends LabeledLineConnectionFigure
        implements DiagramFigure, PropertyChangeListener, SimulatedObjectListener {

    private CERConnection model;
    private static ArrowTip arrowTip = new ArrowTip(15d / 180d * Math.PI, 13, 10, true, false, true);

    private class CardinalityAction extends AbstractAction {

        private int cardinality;

        public CardinalityAction(String name, int cardinality) {
            super(name);
            putValue(ActionUtil.SELECTED_KEY,
                    cardinality == model.getCardinality());
            this.cardinality = cardinality;
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            final int oldValue;
            oldValue = model.getCardinality();
            model.setCardinality(cardinality);

            fireUndoableEditHappened(new AbstractUndoableEdit() {

                @Override
                public String getPresentationName() {
                    ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", Locale.getDefault());
                    return labels.getString("cardinality");
                }

                @Override
                public void undo() throws CannotUndoException {
                    super.undo();
                    model.setCardinality(oldValue);
                }

                @Override
                public void redo() throws CannotUndoException {
                    super.redo();
                    model.setCardinality(cardinality);
                }
            });
        }
    }
    // Relations have no arrow tip
    /*
    private static LineDecoration traversableTip = new ArrowTip(Math.PI / 11, 13, 0, false, true, true);
    private static LineDecoration nonTraversableTip;
    static {
    GeneralPath p = new GeneralPath();
    p.moveTo(-4, 8);
    p.lineTo(4, 16);
    p.moveTo(4,  8);
    p.lineTo(-4, 16);
    nonTraversableTip = new GeneralPathTip(p, 0, false, true, true);
    }*/
    private static LineDecoration traversableTip = null;
    private static LineDecoration nonTraversableTip = null;
    private static LineDecoration connectedWithAttributeTip;

    static {
        Path2D.Double p = new Path2D.Double();
        p.moveTo(-2, 0);
        p.lineTo(2, 0);
        p.lineTo(2, 12);
        p.lineTo(-2, 12);
        p.lineTo(-2, 0);
        connectedWithAttributeTip = new GeneralPathTip(p, 0, true, false, true);
    }

    /**
     * This adapter is used, to connect a TextFigure for an attribute with
     * the SimulatedObject model.
     */
    private class LabelAdapter extends FigureAdapter {

        private boolean owner;

        public LabelAdapter(boolean owner) {
            this.owner = owner;
        }

        @Override
        public void attributeChanged(FigureEvent evt) {
            if (evt.getAttribute().equals(AttributeKeys.TEXT)) {
                String newValue = (String) evt.getNewValue();
                if (newValue == null || newValue.length() == 0) {
                    model.setTraversable(false);
                } else {
                    model.setName(newValue);
                }
            }
        }
    }

    /** Creates a new instance. */
    public CERConnectionFigure() {
        this(CERConnection.MANY_CARDINALITY);
    }

    public CERConnectionFigure(int cardinality) {
        setLayouter(new LocatorLayouter());
        // basicSetLiner(new GERConnectionLiner());
        setLiner(null);
        setModel(new CERConnection(cardinality));
        STROKE_COLOR.set(this, ConceptualERDiagram.CONNECTION_STROKE_COLOR);
        STROKE_WIDTH.set(this, ConceptualERDiagram.DIAGRAM_STROKE_WIDTH);

        setAttributeEnabled(AttributeKeys.END_DECORATION, false);
        setAttributeEnabled(AttributeKeys.START_DECORATION, false);
        setAttributeEnabled(AttributeKeys.STROKE_DASHES, false);
        //  setAttributeEnabled(AttributeKeys.FONT_BOLD, false);
        setAttributeEnabled(AttributeKeys.FONT_ITALIC, false);
        setAttributeEnabled(AttributeKeys.FONT_UNDERLINE, false);
    }

    @Override
    public CERConnection getModel() {
        return model;
    }

    public void setModel(CERConnection m) {
        if (model != null) {
            model.removePropertyChangeListener(this);
            model.removeSimulatedObjectListener(this);
        }
        model = m;
        if (model != null) {
            model.addPropertyChangeListener(this);
            model.addSimulatedObjectListener(this);
        }
        updateLabels();
    }

    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }

    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }

    private void updateLabels() {
        ResourceBundleUtil labels = ConceptualERModel.labels;

        willChange();
        basicRemoveAllChildren();

        setAttributeEnabled(AttributeKeys.START_DECORATION, true);
        setAttributeEnabled(AttributeKeys.END_DECORATION, true);

        if (model == null || model.getStart() == null || model.getEnd() == null) {
            START_DECORATION.set(this, null);
            END_DECORATION.set(this, null);
            setAttributeEnabled(STROKE_TYPE, true);
            STROKE_TYPE.set(this, StrokeType.BASIC);
            setAttributeEnabled(STROKE_TYPE, false);
        } else {
            int concept = model.getSimulatedConcept();
            if (concept == ConceptualERModel.RELATIONSHIP) {
                if (model.getCardinality() == 1) {
                    if (model.getStart().getSimulatedConcept() == ConceptualERModel.ENTITY_SET) {
                        START_DECORATION.set(this, arrowTip);
                    } else {
                        END_DECORATION.set(this, arrowTip);
                    }
                } else {
                    START_DECORATION.set(this, null);
                    END_DECORATION.set(this, null);
                }

                if (model.isTraversable()) {
                    // FIXME - Set label at the end where the entity set is
                    TextFigure roleName = new TextFigure(model.getName());
                    roleName.set(LocatorLayouter.LAYOUT_LOCATOR,
                            new BezierLabelLocator(0.5, -90d / 180d * Math.PI, 2));
                    roleName.addFigureListener(new LabelAdapter(true));
                    roleName.set(FONT_BOLD, true);
                    basicAdd(roleName);
                }
            }
            setAttributeEnabled(STROKE_TYPE, true);
            if (concept == ConceptualERModel.RELATIONSHIP
                    || concept == ConceptualERModel.GENERALIZATION
                    || concept == ConceptualERModel.SPECIALIZATION) {
                // Set double stroke when model isTotal
                STROKE_TYPE.set(this, (model.isTotal()) ? StrokeType.DOUBLE : StrokeType.BASIC);
            } else {
                STROKE_TYPE.set(this, StrokeType.BASIC);
            }
            setAttributeEnabled(STROKE_TYPE, false);
        }

        for (Figure f : getChildren()) {
            AbstractAttributedFigure label = (AbstractAttributedFigure) f;
            label.setAttributeEnabled(STROKE_COLOR, false);
            label.setAttributeEnabled(STROKE_WIDTH, false);
            label.setAttributeEnabled(STROKE_DASHES, false);
            label.setAttributeEnabled(STROKE_MITER_LIMIT, false);
            label.setAttributeEnabled(STROKE_JOIN, false);
            label.setAttributeEnabled(FONT_BOLD, false);
            label.set(TEXT_COLOR, get(TEXT_COLOR));
            label.set(FONT_FACE, get(FONT_FACE));
            label.set(FONT_ITALIC, get(FONT_ITALIC));
            label.set(FONT_UNDERLINE, get(FONT_UNDERLINE));
            label.set(FONT_SIZE, get(FONT_SIZE));
        }

        setAttributeEnabled(START_DECORATION, false);
        setAttributeEnabled(END_DECORATION, false);



        layout();
        changed();
    }

    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;
        final ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.cer.Labels", Locale.getDefault());

        Figure item;

        if (model.getSimulatedConcept() == ConceptualERModel.RELATIONSHIP) {
            action = new CardinalityAction(labels.getString("cardinality1"), 1);
            action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("cardinality"));
            action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinality");
            actions.add(action);

            action = new CardinalityAction(labels.getString("cardinalityM"), CERConnection.MANY_CARDINALITY);
            action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("cardinality"));
            action.putValue(ActionUtil.BUTTON_GROUP_KEY, "cardinality");
            actions.add(action);

            action = new AbstractAction(model.isTraversable() ? labels.getString("relationshipRoleNameHide") : labels.getString("relationshipRoleNameShow")) {

                @Override
                public void actionPerformed(ActionEvent event) {
                    // FIXME - What about undo/redo?
                    model.setTraversable(!model.isTraversable());
                }
            };
            actions.add(action);
        }
        action = new AbstractAction(labels.getString(model.isTotal() ? "relationshipDontMarkAsTotal" : "relationshipMarkAsTotal")) {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setTotal(!model.isTotal());
                fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        return labels.getString("relationship");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        model.setTotal(!model.isTotal());
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.undo();
                        model.setTotal(!model.isTotal());
                    }
                });
            }
        };
        actions.add(action);

        return actions;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        updateLabels();
    }

    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }

    private Simulation getSimulation() {
        return (getDiagram() == null) ? null : getDiagram().getSimulation();
    }

    @Override
    public CERConnectionFigure clone() {
        CERConnectionFigure that = (CERConnectionFigure) super.clone();
        that.setModel(this.getModel().clone());
        return that;
    }

    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    protected void handleDisconnect(Figure start, Figure end) {
        model.setEnd(null);
        model.setStart(null);
        updateLabels();
    }

    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    protected void handleConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure)
                && (end instanceof DiagramFigure)) {
            DiagramFigure sf = (DiagramFigure) start;
            DiagramFigure ef = (DiagramFigure) end;

            model.setEnd((SimulatedElement) ef.getModel());
            model.setStart((SimulatedElement) sf.getModel());

            model.setGeneralization(
                    (getStartConnector() instanceof CERGeneralizationConnector)
                    || (getEndConnector() instanceof CERGeneralizationConnector));

            updateLabels();
        }
    }

    /**
     * Checks if two figures can be connected. Implement this method
     * to constrain the allowed connections between figures.
     */
    public boolean canConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure)
                && end instanceof DiagramFigure) {
            DiagramFigure startdf = (DiagramFigure) start;
            DiagramFigure enddf = (DiagramFigure) end;
            if ((startdf.getModel() instanceof SimulatedElement)
                    && (enddf.getModel() instanceof SimulatedElement)) {
                return model.canConnect(
                        (SimulatedElement) startdf.getModel(),
                        (SimulatedElement) enddf.getModel());
            }
        }
        return false;
    }

    public boolean canConnect(Figure start) {
        if (start instanceof DiagramFigure) {
            DiagramFigure startdf = (DiagramFigure) start;
            if (startdf.getModel() instanceof SimulatedElement) {
                return model.canConnect((SimulatedElement) startdf.getModel());
            }
        }
        return false;
    }

    public int getConnectionCount() {
        return 0;
    }

    public int getConnectionIndex(DiagramFigure f) {
        return 0;
    }

    @Override
    public void read(DOMInput in) throws IOException {
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((CERConnection) in.readObject(0));
        in.closeElement();
        readAttributes(in);
        readPoints(in);
    }

    @Override
    public void write(DOMOutput out) throws IOException {
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
        writePoints(out);
        writeAttributes(out);
    }

    @Override
    public int getLayer() {
        return ConceptualERDiagram.LINK_LAYER;
    }

    @Override
    public void relationshipAdded(SimulatedObjectEvent e) {
    }

    @Override
    public void relationshipRemoved(SimulatedObjectEvent e) {
    }

    @Override
    public void objectChanged(SimulatedObjectEvent e) {
    }

    @Override
    public void objectAdded(SimulatedObjectEvent e) {
    }

    @Override
    public void objectRemoved(SimulatedObjectEvent e) {
        fireFigureRequestRemove();
    }

    @Override
    public void objectRequestRemove(SimulatedObjectEvent e) {
    }
}

